<?php
/*
  Cron job for broadcast (buttons.json support)
  Put this file inside your cron folder and run it regularly.
*/

error_reporting(0);
require_once '../config.php';
define('API_TOKEN', $token);

// connect DB
$db = mysqli_connect('localhost', $databaseUser, $databasePass, $databaseName);

/**
 * BoFile - send request to Telegram API
 * Note: do NOT set Content-Type header manually. Let curl build it for multipart/form-data when needed.
 */
function BoFile($method, $datas = []) {
    $url = "https://api.telegram.org/bot" . API_TOKEN . "/" . $method;
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    // $datas can contain strings (including JSON string for reply_markup) or files
    curl_setopt($ch, CURLOPT_POSTFIELDS, $datas);
    $res = curl_exec($ch);
    if (curl_error($ch)) {
        error_log("BoFile curl error: " . curl_error($ch));
        return null;
    }
    $json = @json_decode($res, true);
    return $json;
}

// ensure lib folder for status files
if (!is_dir(__DIR__ . '/lib')) {
    @mkdir(__DIR__ . '/lib', 0775, true);
}

/**
 * Try to load buttons.json from a few likely locations and return its content (JSON string)
 * The cron script may be in a /cron folder; the main bot may have saved buttons.json in parent folder.
 */
function loadButtonsJson(): ?string {
    $paths = [
        __DIR__ . '/../buttons.json', // common: main bot folder is parent of cron folder
        __DIR__ . '/buttons.json',    // or buttons.json next to cron
        __DIR__ . '/../../buttons.json'
    ];
    foreach ($paths as $p) {
        if (file_exists($p)) {
            $content = trim(file_get_contents($p));
            if ($content !== '') {
                // return raw JSON string (we will pass this as reply_markup)
                return $content;
            }
        }
    }
    return null;
}

// Load inline keyboard JSON (string) if exists
$globalReplyMarkup = loadButtonsJson();

// helper - attach reply_markup (expects $globalReplyMarkup as JSON string)
function withButton(array $params, ?string $globalReplyMarkup): array {
    if ($globalReplyMarkup) {
        $params['reply_markup'] = $globalReplyMarkup;
    }
    return $params;
}

// read send queue (single-row behavior)
$send = mysqli_fetch_assoc(mysqli_query($db, "SELECT * FROM `send` LIMIT 1"));

if (!empty($send['step'])) {

    $offset = (int)$send['user'];
    $batch  = 100;

    // fetch next batch of users
    $usersRes = mysqli_query($db, "SELECT * FROM `user` LIMIT {$batch} OFFSET {$offset}");
    $usersCount = (int)mysqli_num_rows(mysqli_query($db, "SELECT * FROM `user`"));

    // if forward
    if ($send['step'] === 'for') {
        $fromChatId = $send['from'];
        $messageId  = (int)$send['text'];
        while ($row = mysqli_fetch_assoc($usersRes)) {
            $to = $row['from_id'];
            BoFile('forwardMessage', [
                'chat_id'      => $to,
                'from_chat_id' => $fromChatId,
                'message_id'   => $messageId
            ]);
            $db->query("UPDATE `send` SET `user` = `user` + 1 WHERE 1");
        }

    } else {
        // step == 'send' -> payload is JSON stored in send.text
        $payload = json_decode($send['text'], true);
        if (!is_array($payload)) {
            // invalid payload: clear queue
            $db->query("DELETE FROM `send` WHERE 1");
            exit;
        }

        while ($row = mysqli_fetch_assoc($usersRes)) {
            $id = $row['from_id'];

            switch ($payload['type'] ?? '') {
                case 'text':
                    BoFile('sendMessage', withButton([
                        'chat_id' => $id,
                        'text' => $payload['text'] ?? '',
                        'parse_mode' => 'html',
                        'disable_web_page_preview' => true
                    ], $globalReplyMarkup));
                    break;

                case 'photo':
                    BoFile('sendPhoto', withButton([
                        'chat_id' => $id,
                        'photo' => $payload['file_id'],
                        'caption' => $payload['caption'] ?? '',
                        'parse_mode' => 'html'
                    ], $globalReplyMarkup));
                    break;

                case 'video':
                    BoFile('sendVideo', withButton([
                        'chat_id' => $id,
                        'video' => $payload['file_id'],
                        'caption' => $payload['caption'] ?? '',
                        'parse_mode' => 'html'
                    ], $globalReplyMarkup));
                    break;

                case 'audio':
                    BoFile('sendAudio', withButton([
                        'chat_id' => $id,
                        'audio' => $payload['file_id'],
                        'caption' => $payload['caption'] ?? '',
                        'parse_mode' => 'html'
                    ], $globalReplyMarkup));
                    break;

                case 'voice':
                    BoFile('sendVoice', withButton([
                        'chat_id' => $id,
                        'voice' => $payload['file_id'],
                        'caption' => $payload['caption'] ?? '',
                        'parse_mode' => 'html'
                    ], $globalReplyMarkup));
                    break;

                case 'animation':
                    BoFile('sendAnimation', withButton([
                        'chat_id' => $id,
                        'animation' => $payload['file_id'],
                        'caption' => $payload['caption'] ?? '',
                        'parse_mode' => 'html'
                    ], $globalReplyMarkup));
                    break;

                case 'document':
                    BoFile('sendDocument', withButton([
                        'chat_id' => $id,
                        'document' => $payload['file_id'],
                        'caption' => $payload['caption'] ?? '',
                        'parse_mode' => 'html'
                    ], $globalReplyMarkup));
                    break;

                case 'sticker':
                    BoFile('sendSticker', [
                        'chat_id' => $id,
                        'sticker' => $payload['file_id']
                    ]);
                    break;

                case 'video_note':
                    BoFile('sendVideoNote', [
                        'chat_id' => $id,
                        'video_note' => $payload['file_id']
                    ]);
                    break;

                default:
                    // unknown type - skip
                    break;
            }

            // increment progress
            $db->query("UPDATE `send` SET `user` = `user` + 1 WHERE 1");
        }
    }

    // update status message (the admin message)
    $progress = (int)(mysqli_fetch_assoc(mysqli_query($db, "SELECT `user` FROM `send` LIMIT 1"))['user'] ?? 0);
    if ($progress > $usersCount) $progress = $usersCount;

    // read stored status message info
    $statusChatId = @file_get_contents(__DIR__ . '/lib/id.txt');
    $statusMsgId  = @file_get_contents(__DIR__ . '/lib/msg.txt');

    if ($statusChatId && $statusMsgId) {
        BoFile('editMessageReplyMarkup', [
            'chat_id'    => $statusChatId,
            'message_id' => $statusMsgId,
            'reply_markup' => json_encode([
                'inline_keyboard' => [
                    [ ['text' => "📋 ارسال شده: {$progress}", 'callback_data' => 'none'] ],
                    [ ['text' => "⏳ مانده: " . max(0, round(($usersCount - $progress)/100)), 'callback_data' => 'none'] ],
                ]
            ], JSON_UNESCAPED_UNICODE)
        ]);
    }

    // if finished
    if ($progress >= $usersCount) {
        BoFile('sendMessage', [
            'chat_id' => $send['from'],
            'text' => "✅ ارسال برای " . number_format($usersCount) . " کاربر کامل شد."
        ]);
        $db->query("DELETE FROM `send` WHERE 1");
    }

} else {
    echo "There is no message to send\n";
}

// ---------------- delete messages (if present in del table) ----------------
$del = mysqli_query($db, "SELECT * FROM del LIMIT 1000");
if (mysqli_num_rows($del) > 0) {
    while ($row = mysqli_fetch_assoc($del)) {
        BoFile('deleteMessage', [
            'chat_id' => $row['from_id'],
            'message_id' => $row['message_id']
        ]);
        $db->query("DELETE FROM del WHERE message_id = {$row['message_id']} AND from_id = {$row['from_id']}");
    }
    echo "All messages deleted\n";
} else {
    echo "There is no message to delete\n";
}